<?php

namespace Trustcaptcha\Plugins\Elementor;

use Trustcaptcha\Trustcaptcha;
use Trustcaptcha\Plugins\Plugin;
use Elementor\Controls_Stack;
use Elementor\Plugin as ElementorPlugin;
use ElementorPro\Modules\Forms\Classes\Ajax_Handler;
use ElementorPro\Modules\Forms\Classes\Form_Record;

class ElementorForms extends Plugin {

    const FIELD_NAME = 'trustcaptcha';

    public static function get_setup_message()
    {
		return esc_html__( 'To use Trustcaptcha, you need to add your keys in the Trustcaptcha plugin settings.', 'trustcaptcha-for-forms' );
	}

	public static function is_enabled()
    {
		return Trustcaptcha::get_site_key() && Trustcaptcha::get_secret_key();
	}

    public function init(): void
    {
		add_filter( 'elementor_pro/forms/field_types', [ $this, 'add_field_type' ] );
		add_filter( 'elementor_pro/forms/render/item', [ $this, 'filter_field_item' ] );
		add_action( 'elementor_pro/forms/render_field/' . static::FIELD_NAME, [ $this, 'render_field' ], 10, 3 );
        add_action(
			'elementor/element/form/section_form_fields/after_section_end',
			[ $this, 'update_controls' ],
			10,
			2
		);
        add_action( 'wp_print_footer_scripts', [ $this, 'load_scripts' ], 9 );

        if ( static::is_enabled() ) {
			add_action( 'elementor_pro/forms/validation', [ $this, 'validate_captcha' ], 10, 2 );
		}
    }

    public function add_field_type( $field_types )
    {
		$field_types[ static::FIELD_NAME ] = esc_html__( 'Trustcaptcha', 'trustcaptcha-for-forms' );

		return $field_types;
	}

	public function update_controls( Controls_Stack $controls_stack, array $args )
    {
		$control_id   = 'form_fields';
		$control_data = ElementorPlugin::$instance->controls_manager->get_control_from_stack(
			$controls_stack->get_unique_name(),
			$control_id
		);

		$term = [
			'name'     => 'field_type',
			'operator' => '!in',
			'value'    => [ static::FIELD_NAME ],
		];

		$control_data['fields']['width']['conditions']['terms'][]    = $term;
		$control_data['fields']['required']['conditions']['terms'][] = $term;

		ElementorPlugin::$instance->controls_manager->update_control_in_stack(
			$controls_stack,
			$control_id,
			$control_data,
			[ 'recursive' => true ]
		);
	}

	public function render_field( $item, $item_index, $widget )
    {
		$html = '<div class="elementor-field" id="form-field-' . $item['custom_id'] . '">';

		if ( static::is_enabled() ) {
			$this->enqueue_scripts();
			$html .= Trustcaptcha::get_html();
		} elseif ( current_user_can( 'manage_options' ) ) {
			$html .= '<div class="elementor-alert elementor-alert-info">';
			$html .= static::get_setup_message();
			$html .= '</div>';
		}

		$html .= '</div>';

		echo $html;
	}

    public function filter_field_item( $item )
    {
		if ( static::FIELD_NAME === $item['field_type'] ) {
			$item['field_label'] = false;
		}

		return $item;
	}

	public function validate_captcha( $record, $ajax_handler )
    {
		$fields = $record->get_field( [
			'type' => static::FIELD_NAME,
		] );

		if ( empty( $fields ) ) {
			return;
		}
        $field = current( $fields );

        $validationResponse = Trustcaptcha::validateCaptchaWithPostVerificationToken();
        if ( ! $validationResponse->captchaPassed ) {
			$ajax_handler->add_error( $field['id'], __( 'Captcha invalid - '.$validationResponse->errorMessage, 'trustcaptcha-for-forms' ) );
			return;
		}

		$record->remove_field( $field['id'] );
    }

    public function load_scripts()
    {
        wp_enqueue_script(
            'trustcaptcha-elementor',
            constant( 'TRUSTCAPTCHA_BASE_URL' ) . '/assets/js/elementor.js',
            [],
            PLUGIN_VERSION,
            true
		);
    }
}
