<?php

namespace Trustcaptcha\Settings;

class Settings {

    protected $general_tab;
    protected $plugins_tab;
    protected $network_tab;

    public function setup()
    {
		add_action( 'admin_init', [ $this, 'init_admin_settings' ] );
        add_action( 'admin_menu', [ $this, 'add_to_admin_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'add_styles_to_admin_head' ] );
        add_filter( 'admin_footer_text', [ $this, 'admin_footer_text' ] );
        add_filter( 'update_footer', [ $this, 'update_footer' ], PHP_INT_MAX );
        add_filter(
            'plugin_action_links_' . plugin_basename( constant( 'TRUSTCAPTCHA_BASE_FILE' ) ),
            [ $this, 'add_settings_link' ]
        );
        add_filter( 'plugin_row_meta', [ $this, 'add_meta_links' ], 10, 4 );
    }

    public function add_to_admin_menu()
    {
        add_options_page( 'TrustCaptcha', 'TrustCaptcha', 'manage_options', 'trustcaptcha', [ $this, 'init_admin_page' ] );
    }

    public function add_styles_to_admin_head()
    {
		wp_enqueue_style( 'admin', constant( 'TRUSTCAPTCHA_BASE_URL' ) . '/assets/css/settings.css', [], PLUGIN_VERSION, false );
    }

    public function init_admin_settings()
    {
        $this->general_tab = new General();
        $this->plugins_tab = new Plugins();
        $this->network_tab = new Network();

        $this->general_tab->setup();
        $this->plugins_tab->setup();
        $this->network_tab->setup();
    }

    public function init_admin_page()
    {
		if ( ! current_user_can( 'manage_options' ) ) {
            return;
		}

        settings_errors( 'trustcaptcha_messages' );

        $this->render_settings();
    }

    public function render_settings()
    {
		$default_tab = 'general';
        // phpcs:disable WordPress.Security.NonceVerification.Recommended
        $tab = filter_input( INPUT_GET, 'tab', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
        $tab = isset( $tab ) ? $tab : $default_tab;
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
		?>
        <div class="tc-admin">
            <div class="header container">
                <?php printf( '<img src="%s" height="25px"/>', esc_url( constant( 'TRUSTCAPTCHA_BASE_URL' ) . '/assets/img/tc-logo-light.svg' ) ); ?>
                <div class="tc-row">
                    <a class="tc-button" target="_blank" href="https://dashboard.trustcomponent.com/en/dashboard">Dashboard ↗</a>
                    <a class="tc-button" target="_blank" href="https://docs.trustcomponent.com/en/captcha/platforms/wordpress">Documentation ↗</a>
                    <a class="tc-button" target="_blank" href="https://www.trustcomponent.com/en/contact-us">Contact us ↗</a>
                </div>
            </div>
            <nav id="tc-admin-nav" class="container nav-tab-wrapper">
                <a href="?page=trustcaptcha&tab=general" class="nav-tab <?php echo ($tab === 'general' || !$tab) ? 'nav-tab-active' : ''; ?>">General</a>
                <a href="?page=trustcaptcha&tab=plugins" class="nav-tab <?php echo ($tab === 'plugins') ? 'nav-tab-active' : ''; ?>">Plugins</a>
                <a href="?page=trustcaptcha&tab=network" class="nav-tab <?php echo ($tab === 'network') ? 'nav-tab-active' : ''; ?>">Network</a>
            </nav>
            <div class="wrap tab-content">
                <?php
                echo match ($tab) {
                    'general' => $this->general_tab->get_tab_content(),
                    'plugins' => $this->plugins_tab->get_tab_content(),
                    'network' => $this->network_tab->get_tab_content(),
                    default => $this->general_tab->get_tab_content(),
                };
                ?>
            </div>
        </div>
		<?php
    }

    public function add_meta_links( $plugin_meta, $plugin_file, $plugin_data, $status )
    {
        if ( strpos( $plugin_file, plugin_basename( constant( 'TRUSTCAPTCHA_BASE_FILE' ) ) ) !== false ) {
            $new_links = array(
                'doc' => sprintf( '<a href="https://docs.trustcomponent.com/en/captcha/platforms/wordpress" target="_blank">%s</a>', esc_attr( __( 'Documentation', 'trustcaptcha-for-forms' ) ) ),
            );

            $plugin_meta = array_merge( $plugin_meta, $new_links );
        }

        return $plugin_meta;
    }

    public function add_settings_link( array $actions )
    {
        $new_actions = [
            'settings' =>
            '<a href="' . admin_url( 'options-general.php?page=trustcaptcha' ) .
                '" aria-label="' . esc_attr( __( 'Settings', 'trustcaptcha-for-forms' ) ) . '">' .
                esc_html( __( 'Settings', 'trustcaptcha-for-forms' ) ) . '</a>',
        ];

        return array_merge( $new_actions, $actions );
    }

    private function is_visible()
    {
        $current_screen = get_current_screen()->id;
        return 'settings_page_trustcaptcha' === $current_screen;
    }

    public function update_footer( $content )
    {
        if ( ! $this->is_visible() ) {
            return $content;
        }

        return sprintf(
        __( 'Version %s', 'trustcaptcha-for-forms' ),
        PLUGIN_VERSION
        );
	}

    public function admin_footer_text( $text )
    {
        if ( ! $this->is_visible() ) {
            return $text;
        }

        return '';
    }
}
