<?php

namespace Trustcaptcha;

use TrustComponent\TrustCaptcha\CaptchaManager;
use TrustComponent\TrustCaptcha\SecretKeyInvalidException;

class Trustcaptcha {

    public static function get_styles()
    {
        ?>
        <style>
            .wpforms-container .trustcaptcha,
            .trustcaptcha {
                margin-bottom: 16px;
            }

            .trustcaptcha[data-invisible="true"] {
                margin-bottom: 0;
            }
        </style>
		<?php
    }

    public static function load_head()
    {
		wp_enqueue_script( 'trustcaptcha-form', constant( 'TRUSTCAPTCHA_BASE_URL' ) . '/assets/js/form.js', [], PLUGIN_VERSION, true );
        wp_enqueue_script( 'trustcaptcha', self::get_script(), [], PLUGIN_VERSION, true );
        self::get_styles();
    }

    public static function get_script()
    {
		return 'https://cdn.trustcomponent.com/trustcaptcha/2.0.x/trustcaptcha.umd.min.js';
    }

    public static function get_ob_html()
    {
		ob_start();
        self::get_html();

        return ob_get_clean();
    }

    public static function get_html() {
        $allowed_tags = array_merge(
            wp_kses_allowed_html( 'post' ),
            array(
                'trustcaptcha-component' => array(
                    'sitekey' => true,
                    'license' => true,
                    'width' => true,
                    'language' => true,
                    'theme' => true,
                    'autostart' => true,
                    'privacy-url' => true,
                    'hide-branding' => true,
                    'invisible' => true,
                    'invisible-hint' => true,
                    'mode' => true,
                    'custom-translations' => true,
                    'custom-design' => true,
                    'framework' => true,
                ),
            )
        );
        print wp_kses( self::build_html(), $allowed_tags );
    }

    public static function build_html( $options = null ): string
    {
        if ( ! $options ) {
            $options = self::get_saved_widget_options();
        }

        return '<div class="trustcaptcha" data-invisible="false">
                   <trustcaptcha-component 
                        sitekey="'.$options['site_key'].'" 
                        license="'.$options['license_key'].'"
                        width="'.$options['width'].'"
                        language="'.$options['language'].'"
                        theme="'.$options['theme'].'"
                        autostart="'.( ! empty( $options['autostart'] ) ? 'true' : 'false' ).'"
                        privacy-url="'.$options['privacy_url'].'"
                        hide-branding="'.( ! empty( $options['hide_branding'] ) ? 'true' : 'false' ).'"
                        invisible="'.( ! empty( $options['invisible'] ) ? 'true' : 'false' ).'"
                        invisible-hint="'.$options['invisible_hint'].'"
                        mode="'.$options['mode'].'"
                        custom-translations=\''.$options['custom_translations'].'\'
                        custom-design=\''.$options['custom_design'].'\'
                        framework="wordpress"
                    ></trustcaptcha-component>
                </div>';
    }

    private static function get_saved_widget_options(): array
    {
        $options = self::get_saved_options();
        return [
            'site_key'      => $options['site_key'],
            'secret_key'    => $options['secret_key'],
            'license_key'   => $options['license_key'],
            'threshold'     => $options['threshold'],
            'width'         => $options['width'],
            'language'      => $options['language'],
            'theme'         => $options['theme'],
            'autostart'     => $options['autostart'],
            'privacy_url'   => $options['privacy_url'],
            'hide_branding' => $options['hide_branding'],
            'invisible'     => $options['invisible'],
            'invisible_hint' => $options['invisible_hint'],
            'mode'          => $options['mode'],
            'custom_translations' => $options['custom_translations'],
            'custom_design' => $options['custom_design'],
        ];
    }

    private static function get_saved_options(): array
    {
        $options = get_option( 'trustcaptcha_options' );
        return [
            'site_key'      => $options['field_site_key'] ?? '',
            'secret_key'    => $options['field_secret_key'] ?? '',
            'license_key'   => $options['field_license_key'] ?? '',
            'threshold'     => $options['field_threshold'] ?? 0.5,
            'width'         => $options['field_width'] ?? 'fixed',
            'language'      => $options['field_language'] ?? 'auto',
            'theme'         => $options['field_theme'] ?? 'light',
            'autostart'     => $options['field_autostart'] ?? true,
            'privacy_url'   => $options['field_privacy_url'] ?? '',
            'hide_branding' => $options['field_hide_branding'] ?? false,
            'invisible'     => $options['field_invisible'] ?? false,
            'invisible_hint' => $options['field_invisible_hint'] ?? 'right-border',
            'mode'          => $options['field_mode'] ?? 'standard',
            'custom_translations' => $options['field_custom_translations'] ?? '',
            'custom_design' => $options['field_custom_design'] ?? '',
        ];
    }

    public static function get_site_key(): string
    {
        return self::get_saved_options()['site_key'];
    }

    public static function get_secret_key(): string
    {
        return self::get_saved_options()['secret_key'];
    }

    public static function get_threshold(): string
    {
        return self::get_saved_options()['threshold'];
    }

    public static function validateCaptchaWithPostVerificationToken(): ValidationResponse
    {

        if ( ! isset( $_POST['tc-verification-token'] ) ) {
            return new ValidationResponse(false, 'CAPTCHA has not yet been completed. Please wait until the CAPTCHA has been passed.');
        }

        $verification_token = filter_var( wp_unslash( $_POST['tc-verification-token'] ), FILTER_SANITIZE_FULL_SPECIAL_CHARS );

        return self::validateCaptcha( $verification_token );
    }

    public static function validateCaptcha(string $verification_token): ValidationResponse
    {
        $verification_token = sanitize_text_field($verification_token);
        $secret_key = self::get_secret_key();
        $threshold = self::get_threshold();

        if (empty($verification_token)) {
            return new ValidationResponse(false, 'CAPTCHA has not yet been completed. Please wait until the CAPTCHA has been passed.');
        }

        $network_options = get_option('trustcaptcha_network_options');
        $proxyOptions = null;
        if (!empty($network_options['field_enable_proxy'])) {
            $proxy_host = $network_options['field_proxy_host'] ?? '';
            if (!empty($proxy_host)) {
                $proxyOptions = ['proxy' => $proxy_host];
                $proxy_username = $network_options['field_proxy_username'] ?? '';
                $proxy_password = $network_options['field_proxy_password'] ?? '';
                if (!empty($proxy_username) && !empty($proxy_password)) {
                    $proxyOptions['username'] = $proxy_username;
                    $proxyOptions['password'] = $proxy_password;
                }
            }
        }

        try {
            $verificationResult = CaptchaManager::getVerificationResult($secret_key, $verification_token, $proxyOptions);
        } catch (SecretKeyInvalidException|\Exception $e) {
            return new ValidationResponse(false, 'CAPTCHA verification failed due to an error. Exception: ' . $e->getMessage());
        }

        if (!$verificationResult->verificationPassed || $verificationResult->score > $threshold) {
            error_log('TrustCaptcha CAPTCHA not passed or the bot-score ' . $verificationResult->score . ' is higher than your set threshold of ' . $threshold);
            return new ValidationResponse(false, 'Action canceled because we could not be sure that you are actually a human user. Please try again later.');
        }

        return new ValidationResponse(true, '');
    }
}

class ValidationResponse
{
    public bool $captchaPassed;
    public string $errorMessage;

    public function __construct(bool $captchaPassed, string $errorMessage)
    {
        $this->captchaPassed = $captchaPassed;
        $this->errorMessage = $errorMessage;
    }
}
